function hess = LWM_HessianR(parm,M1,M2,resid,Q_grad,delta,vega,Gamma,eta,xi,weights)
%==========================================================================================
% This function computes the constraint-reduced Hessian matrix for the constraint-reduced/free
% parameter vector.
%
% INPUT:
%        parm         : 3(M1+M2)-by-1 vector of estimated parameters (full version)
%        M1           : number of Lognormal densities in the mixture
%        M2           : number of Weibull densities in the mixture
%        resid        : N-by-1 vector of option residuals
%        Q_grad       : N-by-3M Jacobian matrix: dO(theta)/dtheta
%        delta        : N-by-M matrix of component-wise option delta
%        vega         : N-by-M matrix of component-wise option vega
%        gamma        : N-by-M matrix of component-wise option gamma
%        eta          : N-by-M matrix of component-wise option eta, where eta is defined as d^2O(theta)/dsigma^2
%        xi           : N-by-M matrix of component-wise option xi, where
%                       xi is defined as d^2O(theta)/dsigma/dF
%        weights      : N-by-1 vector of option weights. Default: equal weights
%
%
% OUTPUT:
%       hess           : (3M-2)-by-(3M-2) Hessian matrix defined as
%                         d^2Q(theta)/dtheta/dtheta'.
%==========================================================================================
% This ver: 2023/05/24
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2023). Parametric Risk-Neutral 
%          Density Estimation via Finite Lognormal-Weibull Mixtures
%========================================================================================== 

K=length(parm)-2;
W = diag(weights);
M = M1 + M2;
%Isolate weight and component-wise F
w_vec=parm(1:M)';
f_vec=parm(M+1:2*M)';
% sig_vec = parm(2*M+1:3*M);


%Calculate the "outerproduct of gradient part" of Hessian
hess=Q_grad'*W*Q_grad;
%Determine the range of the parameters in the parameter vector
w_range = 1:M-1;
f_range = M:2*M-2;
sig_range = 2*M-1:3*M-2;


%Initializing the "observation-wise Hessian" part of Hessian
hmat=zeros(K,K);
%Calculate the second derivative w.r.t. sigma_i^2
wresid = W*resid; %the weighted residual
hmat(sig_range,sig_range)=diag((w_vec.*xi)'*wresid);
if M>1
    %Calculate the second derivatives w.r.t. w_i  w_j
    hmat(w_range,w_range)= ivech((Gamma(:,end)/w_vec(end)*vech((f_vec(end)-f_vec(1:end-1))'*(f_vec(end)-f_vec(1:end-1)))')'*wresid);
    %Calcualte the second derivative w.r.t. F_i F_j
    hmat(f_range,f_range)=ivech((Gamma(:,end)*vech(w_vec(1:end-1)'*w_vec(1:end-1)/w_vec(end))')'*wresid)+diag((w_vec(1:end-1).*Gamma(:,1:end-1))'*wresid);
    %Calculate the second derivative w.r.t. sigma_i w_j for i<M
    hmat(w_range,sig_range(1:end-1))=diag(vega(:,1:end-1)'*wresid);
    %Calculate the second derivative w.r.t. sigma_i F_j for i<M
    hmat(f_range,sig_range(1:end-1))=diag((w_vec(1:end-1).*eta(:,1:end-1))'*wresid);
    %Calculate the second derivative w.r.t. sigma_M w_i
    hmat(w_range,sig_range(end))=((f_vec(end)-f_vec(1:end-1)).*eta(:,end)-vega(:,end))'*wresid;
    %Calculate the second derivative w.r.t. sigma_M F_i
    hmat(f_range,sig_range(end))=-(w_vec(1:end-1).*eta(:,end))'*wresid;
    %Calculate the second derivative w.r.t. F_i w_j
    hmat(f_range,w_range)=reshape((Gamma(:,end)*reshape(-repmat(w_vec(1:end-1),M-1,1)'/w_vec(end).*(f_vec(end)-repmat(f_vec(1:end-1),M-1,1)),1,(M-1)^2))'*wresid,M-1,M-1) ...
        +diag((delta(:,1:end-1)-delta(:,end))'*wresid);
end
%Replicate the lower triangular half to form a symmetric matrix
hmat=ivech(vech(hmat));
%Combine the two parts of Hessian
hess=-hmat+hess;
end


%========================================================================
%Utility functions from MFE toolbox of Kevin Sheppard
%========================================================================

function stackedData = vech(matrixData)
[k,~] = size(matrixData);

sel = tril(true(k));
stackedData = matrixData(sel);
end
function matrixData=ivech(stackedData)

if size(stackedData,2)>size(stackedData,1)
    stackedData=stackedData';
end

if size(stackedData,2)~=1
    error('STACKED_DATA must be a column vector.')
end

K2=size(stackedData,1);
K=(-1+sqrt(1+8*K2))/2;
if floor(K)~=K
    error(['The number of elemeents in STACKED_DATA must be conformable to' ...
        'the inverse vech operation.'])
end

matrixData=zeros(K);
pl=tril(true(K));
matrixData(pl)=stackedData;
diag_matrixData=diag(diag(matrixData));
matrixData=matrixData+matrixData'-diag_matrixData;
end
